// Web Design Ruler Content Script
console.log('Web Design Ruler content script loaded:', document.location.href);

// Message listener for all communications
chrome.runtime.onMessage.addListener(function(message, sender, sendResponse) {
  console.log('Content script received message:', message.action);
  
  // Respond to ping to confirm content script is loaded
  if (message.action === "ping") {
    console.log('Ping received, responding with pong');
    sendResponse({pong: true});
    return true;
  }
  
  // Handle color picker activation
  if (message.action === "activateColorPicker") {
    console.log('Activating color picker tool');
    activateEyedropper();
    sendResponse({success: true});
    return true;
  }
  
  // Handle font detector activation
  if (message.action === "activateFontDetector") {
    console.log('Activating font detector tool');
    activateFontDetector();
    sendResponse({success: true});
    return true;
  }
  
  // Handle measurement tool activation
  if (message.action === "activateMeasureTool") {
    console.log('Activating measurement tool');
    activateMeasureTool();
    sendResponse({success: true});
    return true;
  }
  
  return true; // Keep the message channel open for async responses
});

// Eyedropper tool implementation
function activateEyedropper() {
  console.log('Eyedropper function started');
  
  // Store original styles to restore later
  const originalCursor = document.body.style.cursor;
  const originalUserSelect = document.body.style.userSelect;
  
  // Change cursor to indicate color picking mode
  document.body.style.cursor = 'crosshair';
  document.body.style.userSelect = 'none';
  
  // Create color display element
  const colorDisplay = document.createElement('div');
  colorDisplay.style.position = 'fixed';
  colorDisplay.style.bottom = '20px';
  colorDisplay.style.left = '50%';
  colorDisplay.style.transform = 'translateX(-50%)';
  colorDisplay.style.backgroundColor = 'rgba(0, 0, 0, 0.9)';
  colorDisplay.style.color = 'white';
  colorDisplay.style.padding = '12px 20px';
  colorDisplay.style.borderRadius = '6px';
  colorDisplay.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  colorDisplay.style.fontSize = '14px';
  colorDisplay.style.zIndex = '2147483647'; // Maximum z-index
  colorDisplay.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.3)';
  colorDisplay.style.display = 'flex';
  colorDisplay.style.alignItems = 'center';
  colorDisplay.style.gap = '12px';
  colorDisplay.innerHTML = '<span style="font-size: 16px;">🎨</span><span>Hover over an element to see its color. Click to pick.</span>';
  document.body.appendChild(colorDisplay);
  
  console.log('Color display element created');
  
  // Flag to track if we're still in color picking mode
  let isPickerActive = true;
  let currentColor = null;
  
  // Mouse move handler
  function onMouseMove(e) {
    if (!isPickerActive) return;
    
    try {
      const element = document.elementFromPoint(e.clientX, e.clientY);
      
      if (element && element !== colorDisplay) {
        // Get computed style
        let computedStyle = window.getComputedStyle(element);
        let bgColor = computedStyle.backgroundColor;
        
        // If the background is transparent, try parent elements
        let targetElement = element;
        while ((bgColor === 'rgba(0, 0, 0, 0)' || bgColor === 'transparent') && targetElement.parentElement) {
          targetElement = targetElement.parentElement;
          computedStyle = window.getComputedStyle(targetElement);
          bgColor = computedStyle.backgroundColor;
        }
        
        // Convert RGB to HEX
        currentColor = rgbToHex(bgColor);
        
        // Update color display
        colorDisplay.innerHTML = `<span style="display:inline-block;width:24px;height:24px;background-color:${currentColor};border:2px solid white;border-radius:4px;"></span><span style="font-family:monospace;font-size:16px;font-weight:600;">${currentColor}</span>`;
        
        console.log('Hovering over element with color:', currentColor);
      }
    } catch (error) {
      console.error('Error in mousemove handler:', error);
    }
  }
  
  // Click handler
  function onClick(e) {
    if (!isPickerActive) return;
    
    // Prevent default behavior (like following links)
    e.preventDefault();
    e.stopPropagation();
    
    if (currentColor) {
      console.log('Color picked:', currentColor);
      
      // Save the color
      chrome.storage.local.set({lastPickedColor: currentColor}, function() {
        console.log('Color saved to storage:', currentColor);
      });
      
      // Send message to background
      chrome.runtime.sendMessage({
        action: 'colorPicked',
        color: currentColor
      });
      
      // Show a confirmation
      colorDisplay.innerHTML = `<span style="color:#10B981;">✓</span> <span>Color picked: ${currentColor}</span>`;
      
      // Clean up after a short delay
      setTimeout(cleanup, 1000);
    }
  }
  
  // Escape key handler
  function onKeyDown(e) {
    if (e.key === 'Escape' && isPickerActive) {
      console.log('Eyedropper cancelled with Escape key');
      cleanup();
    }
  }
  
  // Cleanup function
  function cleanup() {
    if (!isPickerActive) return;
    
    console.log('Cleaning up eyedropper');
    isPickerActive = false;
    
    // Remove event listeners
    document.removeEventListener('mousemove', onMouseMove);
    document.removeEventListener('click', onClick);
    document.removeEventListener('keydown', onKeyDown);
    
    // Restore original styles
    document.body.style.cursor = originalCursor;
    document.body.style.userSelect = originalUserSelect;
    
    // Remove color display
    if (colorDisplay && colorDisplay.parentNode) {
      document.body.removeChild(colorDisplay);
    }
    
    console.log('Eyedropper cleanup complete');
  }
  
  // Add event listeners
  document.addEventListener('mousemove', onMouseMove);
  document.addEventListener('click', onClick);
  document.addEventListener('keydown', onKeyDown);
  
  console.log('Eyedropper event listeners added');
}

// Font detector tool implementation
function activateFontDetector() {
  console.log('Font detector function started');
  
  // Store original styles to restore later
  const originalCursor = document.body.style.cursor;
  const originalUserSelect = document.body.style.userSelect;
  
  // Change cursor to indicate font detection mode
  document.body.style.cursor = 'text';
  document.body.style.userSelect = 'none';
  
  // Create overlay element
  const overlay = document.createElement('div');
  overlay.style.position = 'fixed';
  overlay.style.top = '0';
  overlay.style.left = '0';
  overlay.style.width = '100%';
  overlay.style.height = '100%';
  overlay.style.zIndex = '2147483646'; // One less than max (for info panel)
  overlay.style.pointerEvents = 'none'; // Allow clicks to pass through
  document.body.appendChild(overlay);
  
  // Create info panel
  const infoPanel = document.createElement('div');
  infoPanel.style.position = 'fixed';
  infoPanel.style.bottom = '20px';
  infoPanel.style.left = '50%';
  infoPanel.style.transform = 'translateX(-50%)';
  infoPanel.style.backgroundColor = 'rgba(0, 0, 0, 0.9)';
  infoPanel.style.color = 'white';
  infoPanel.style.padding = '16px 20px';
  infoPanel.style.borderRadius = '6px';
  infoPanel.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  infoPanel.style.fontSize = '13px';
  infoPanel.style.zIndex = '2147483647'; // Maximum z-index
  infoPanel.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.3)';
  infoPanel.style.width = '320px';
  infoPanel.style.maxWidth = '90%';
  infoPanel.innerHTML = `
    <div style="display:flex;align-items:center;gap:8px;margin-bottom:12px;">
      <span style="font-size:16px;">🔤</span>
      <span style="font-weight:600;">Font Detector</span>
    </div>
    <div style="color:#9CA3AF;">Hover over text to see font details. Click to select.</div>
  `;
  document.body.appendChild(infoPanel);
  
  console.log('Font detector elements created');
  
  // Track current element and highlighting
  let isActive = true;
  let highlightedElement = null;
  let highlightOverlay = null;
  
  // Create highlight overlay function
  function createHighlightOverlay(element) {
    const rect = element.getBoundingClientRect();
    
    // Remove previous highlight if exists
    if (highlightOverlay) {
      document.body.removeChild(highlightOverlay);
    }
    
    // Create new highlight
    highlightOverlay = document.createElement('div');
    highlightOverlay.style.position = 'fixed';
    highlightOverlay.style.top = rect.top + 'px';
    highlightOverlay.style.left = rect.left + 'px';
    highlightOverlay.style.width = rect.width + 'px';
    highlightOverlay.style.height = rect.height + 'px';
    highlightOverlay.style.border = '2px solid #2563EB';
    highlightOverlay.style.backgroundColor = 'rgba(37, 99, 235, 0.1)';
    highlightOverlay.style.zIndex = '2147483645'; // Below overlay
    highlightOverlay.style.pointerEvents = 'none';
    highlightOverlay.style.borderRadius = '4px';
    document.body.appendChild(highlightOverlay);
    
    return highlightOverlay;
  }
  
  // Get font details function
  function getFontDetails(element) {
    const computedStyle = window.getComputedStyle(element);
    
    // Extract font properties
    const fontFamily = computedStyle.fontFamily;
    const fontSize = computedStyle.fontSize;
    const fontWeight = computedStyle.fontWeight;
    const fontStyle = computedStyle.fontStyle;
    const lineHeight = computedStyle.lineHeight;
    const letterSpacing = computedStyle.letterSpacing;
    const color = computedStyle.color;
    const textTransform = computedStyle.textTransform;
    
    // Handle font weight names
    let fontWeightName = fontWeight;
    const weightMap = {
      '100': '100 (Thin)',
      '200': '200 (Extra Light)',
      '300': '300 (Light)',
      '400': '400 (Normal)',
      '500': '500 (Medium)',
      '600': '600 (Semi Bold)',
      '700': '700 (Bold)',
      '800': '800 (Extra Bold)',
      '900': '900 (Black)'
    };
    fontWeightName = weightMap[fontWeight] || fontWeight;
    
    // Format color as hex
    const colorHex = rgbToHex(color);
    
    // Create CSS for this font
    const fontCSS = `font-family: ${fontFamily};
font-size: ${fontSize};
font-weight: ${fontWeight};
font-style: ${fontStyle};
line-height: ${lineHeight};
letter-spacing: ${letterSpacing};
color: ${colorHex};
text-transform: ${textTransform};`;
    
    return {
      fontFamily,
      fontSize, 
      fontWeight: fontWeightName,
      fontStyle,
      lineHeight,
      letterSpacing,
      color: colorHex,
      textTransform,
      css: fontCSS
    };
  }
  
  // Update info panel with font details
  function updateInfoPanel(element) {
    const fontDetails = getFontDetails(element);
    
    // Format family for display (remove quotes)
    let displayFamily = fontDetails.fontFamily.replace(/['"]/g, '');
    if (displayFamily.length > 30) {
      displayFamily = displayFamily.substring(0, 27) + '...';
    }
    
    infoPanel.innerHTML = `
      <div style="display:flex;align-items:center;gap:8px;margin-bottom:12px;">
        <span style="font-size:16px;">🔤</span>
        <span style="font-weight:600;">Font Detector</span>
      </div>
      <div style="margin-bottom:12px;padding:8px;background:rgba(255,255,255,0.1);border-radius:4px;">
        <div style="font-family:${fontDetails.fontFamily};font-size:14px;margin-bottom:4px;">${displayFamily}</div>
        <div style="font-size:11px;color:#9CA3AF;">Sample Text</div>
      </div>
      <div style="display:grid;grid-template-columns:auto 1fr;gap:8px;font-size:12px;">
        <span style="color:#9CA3AF;">Size:</span>
        <span>${fontDetails.fontSize}</span>
        <span style="color:#9CA3AF;">Weight:</span>
        <span>${fontDetails.fontWeight}</span>
        <span style="color:#9CA3AF;">Color:</span>
        <span style="display:flex;align-items:center;gap:6px;">
          ${fontDetails.color}
          <span style="display:inline-block;width:14px;height:14px;background-color:${fontDetails.color};border:1px solid rgba(255,255,255,0.3);border-radius:2px;"></span>
        </span>
      </div>
      <div style="font-size:11px;color:#9CA3AF;margin-top:12px;text-align:center;">Click to select this font</div>
    `;
  }
  
  // Mouse move handler
  function onMouseMove(e) {
    if (!isActive) return;
    
    try {
      const element = document.elementFromPoint(e.clientX, e.clientY);
      
      // Skip overlay elements
      if (!element || element === overlay || element === infoPanel || element === highlightOverlay) {
        return;
      }
      
      // Skip if element hasn't changed
      if (element === highlightedElement) {
        return;
      }
      
      // Update highlighted element
      highlightedElement = element;
      createHighlightOverlay(element);
      updateInfoPanel(element);
      
    } catch (error) {
      console.error('Error in font detector mousemove handler:', error);
    }
  }
  
  // Click handler
  function onClick(e) {
    if (!isActive) return;
    
    // Prevent default to avoid following links
    e.preventDefault();
    e.stopPropagation();
    
    try {
      const element = document.elementFromPoint(e.clientX, e.clientY);
      
      // Skip overlay elements
      if (!element || element === overlay || element === infoPanel || element === highlightOverlay) {
        return;
      }
      
      // Get font details
      const fontDetails = getFontDetails(element);
      console.log('Font detected:', fontDetails);
      
      // Save to storage
      chrome.storage.local.set({lastDetectedFont: fontDetails}, function() {
        console.log('Font details saved to storage');
      });
      
      // Send message to background
      chrome.runtime.sendMessage({
        action: 'fontDetected',
        fontDetails: fontDetails
      });
      
      // Show confirmation
      infoPanel.innerHTML = `
        <div style="text-align:center;padding:20px;">
          <div style="color:#10B981;font-size:24px;margin-bottom:8px;">✓</div>
          <div style="font-weight:600;margin-bottom:4px;">Font Selected!</div>
          <div style="font-family:${fontDetails.fontFamily};font-size:14px;color:#9CA3AF;">${fontDetails.fontFamily.replace(/['"]/g, '')}</div>
        </div>
      `;
      
      // Clean up after delay
      setTimeout(cleanup, 1500);
      
    } catch (error) {
      console.error('Error in font detector click handler:', error);
    }
  }
  
  // Escape key handler
  function onKeyDown(e) {
    if (e.key === 'Escape' && isActive) {
      console.log('Font detector cancelled with Escape key');
      cleanup();
    }
  }
  
  // Cleanup function
  function cleanup() {
    if (!isActive) return;
    
    console.log('Cleaning up font detector');
    isActive = false;
    
    // Remove event listeners
    document.removeEventListener('mousemove', onMouseMove);
    document.removeEventListener('click', onClick);
    document.removeEventListener('keydown', onKeyDown);
    
    // Restore original styles
    document.body.style.cursor = originalCursor;
    document.body.style.userSelect = originalUserSelect;
    
    // Remove overlay elements
    if (overlay && overlay.parentNode) {
      document.body.removeChild(overlay);
    }
    
    if (infoPanel && infoPanel.parentNode) {
      document.body.removeChild(infoPanel);
    }
    
    if (highlightOverlay && highlightOverlay.parentNode) {
      document.body.removeChild(highlightOverlay);
    }
    
    console.log('Font detector cleanup complete');
  }
  
  // Add event listeners
  document.addEventListener('mousemove', onMouseMove);
  document.addEventListener('click', onClick);
  document.addEventListener('keydown', onKeyDown);
  
  console.log('Font detector event listeners added');
}

// Measurement tool implementation
function activateMeasureTool() {
  console.log('Measurement tool function started');
  
  // Store original styles to restore later
  const originalCursor = document.body.style.cursor;
  const originalUserSelect = document.body.style.userSelect;
  
  // Change cursor to indicate measurement mode
  document.body.style.cursor = 'crosshair';
  document.body.style.userSelect = 'none';
  
  // Create UI elements
  const overlay = document.createElement('div');
  overlay.style.position = 'fixed';
  overlay.style.top = '0';
  overlay.style.left = '0';
  overlay.style.width = '100%';
  overlay.style.height = '100%';
  overlay.style.zIndex = '2147483646'; // One less than max
  overlay.style.pointerEvents = 'auto'; // Capture clicks
  overlay.style.cursor = 'crosshair';
  document.body.appendChild(overlay);
  
  // Create info panel
  const infoPanel = document.createElement('div');
  infoPanel.style.position = 'fixed';
  infoPanel.style.bottom = '20px';
  infoPanel.style.left = '50%';
  infoPanel.style.transform = 'translateX(-50%)';
  infoPanel.style.backgroundColor = 'rgba(0, 0, 0, 0.9)';
  infoPanel.style.color = 'white';
  infoPanel.style.padding = '16px 20px';
  infoPanel.style.borderRadius = '6px';
  infoPanel.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  infoPanel.style.fontSize = '13px';
  infoPanel.style.zIndex = '2147483647'; // Maximum z-index
  infoPanel.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.3)';
  infoPanel.style.width = '280px';
  infoPanel.style.maxWidth = '90%';
  infoPanel.innerHTML = `
    <div style="display:flex;align-items:center;gap:8px;margin-bottom:8px;">
      <span style="font-size:16px;">📐</span>
      <span style="font-weight:600;">Measurement Tool</span>
    </div>
    <div style="color:#9CA3AF;">Click and drag to measure distances</div>
  `;
  document.body.appendChild(infoPanel);
  
  // Create measurement box
  const measureBox = document.createElement('div');
  measureBox.style.position = 'fixed';
  measureBox.style.border = '1px dashed #2563EB';
  measureBox.style.backgroundColor = 'rgba(37, 99, 235, 0.1)';
  measureBox.style.zIndex = '2147483645';
  measureBox.style.display = 'none';
  measureBox.style.pointerEvents = 'none';
  document.body.appendChild(measureBox);
  
  // Create dimension labels
  const widthLabel = document.createElement('div');
  widthLabel.style.position = 'fixed';
  widthLabel.style.backgroundColor = '#2563EB';
  widthLabel.style.color = 'white';
  widthLabel.style.padding = '2px 8px';
  widthLabel.style.borderRadius = '3px';
  widthLabel.style.fontSize = '12px';
  widthLabel.style.fontWeight = '600';
  widthLabel.style.zIndex = '2147483647';
  widthLabel.style.pointerEvents = 'none';
  widthLabel.style.display = 'none';
  widthLabel.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  document.body.appendChild(widthLabel);
  
  const heightLabel = document.createElement('div');
  heightLabel.style.position = 'fixed';
  heightLabel.style.backgroundColor = '#2563EB';
  heightLabel.style.color = 'white';
  heightLabel.style.padding = '2px 8px';
  heightLabel.style.borderRadius = '3px';
  heightLabel.style.fontSize = '12px';
  heightLabel.style.fontWeight = '600';
  heightLabel.style.zIndex = '2147483647';
  heightLabel.style.pointerEvents = 'none';
  heightLabel.style.display = 'none';
  heightLabel.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  document.body.appendChild(heightLabel);
  
  const diagonalLabel = document.createElement('div');
  diagonalLabel.style.position = 'fixed';
  diagonalLabel.style.backgroundColor = '#10B981';
  diagonalLabel.style.color = 'white';
  diagonalLabel.style.padding = '2px 8px';
  diagonalLabel.style.borderRadius = '3px';
  diagonalLabel.style.fontSize = '12px';
  diagonalLabel.style.fontWeight = '600';
  diagonalLabel.style.zIndex = '2147483647';
  diagonalLabel.style.pointerEvents = 'none';
  diagonalLabel.style.display = 'none';
  diagonalLabel.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
  document.body.appendChild(diagonalLabel);
  
  console.log('Measurement tool elements created');
  
  // Track measurement state
  let isActive = true;
  let isDrawing = false;
  let startX = 0;
  let startY = 0;
  let endX = 0;
  let endY = 0;
  
  // Update measurement display
  function updateMeasurement(currentX, currentY) {
    if (!isDrawing) return;
    
    // Update end coordinates
    endX = currentX;
    endY = currentY;
    
    // Calculate dimensions
    const width = Math.abs(endX - startX);
    const height = Math.abs(endY - startY);
    const diagonal = Math.sqrt(width * width + height * height);
    
    // Calculate positions
    const boxLeft = Math.min(startX, endX);
    const boxTop = Math.min(startY, endY);
    
    // Update measurement box
    measureBox.style.display = 'block';
    measureBox.style.left = boxLeft + 'px';
    measureBox.style.top = boxTop + 'px';
    measureBox.style.width = width + 'px';
    measureBox.style.height = height + 'px';
    
    // Update labels
    widthLabel.style.display = 'block';
    widthLabel.textContent = width + 'px';
    widthLabel.style.left = (boxLeft + width / 2 - widthLabel.offsetWidth / 2) + 'px';
    widthLabel.style.top = (boxTop - widthLabel.offsetHeight - 5) + 'px';
    
    heightLabel.style.display = 'block';
    heightLabel.textContent = height + 'px';
    heightLabel.style.left = (boxLeft - heightLabel.offsetWidth - 5) + 'px';
    heightLabel.style.top = (boxTop + height / 2 - heightLabel.offsetHeight / 2) + 'px';
    
    diagonalLabel.style.display = 'block';
    diagonalLabel.textContent = Math.round(diagonal) + 'px';
    diagonalLabel.style.left = (boxLeft + width / 2 - diagonalLabel.offsetWidth / 2) + 'px';
    diagonalLabel.style.top = (boxTop + height / 2 - diagonalLabel.offsetHeight / 2) + 'px';
    
    // Update info panel
    infoPanel.innerHTML = `
      <div style="display:flex;align-items:center;gap:8px;margin-bottom:12px;">
        <span style="font-size:16px;">📐</span>
        <span style="font-weight:600;">Measurement Tool</span>
      </div>
      <div style="display:grid;grid-template-columns:auto 1fr;gap:8px;font-size:12px;">
        <span style="color:#9CA3AF;">Width:</span>
        <span style="font-weight:600;">${width}px</span>
        <span style="color:#9CA3AF;">Height:</span>
        <span style="font-weight:600;">${height}px</span>
        <span style="color:#9CA3AF;">Diagonal:</span>
        <span style="font-weight:600;">${Math.round(diagonal)}px</span>
      </div>
      <div style="font-size:11px;color:#9CA3AF;margin-top:12px;text-align:center;">Release to save, ESC to cancel</div>
    `;
  }
  
  // Mouse down handler
  function onMouseDown(e) {
    if (!isActive || isDrawing) return;
    
    // Start drawing
    isDrawing = true;
    startX = e.clientX;
    startY = e.clientY;
    endX = e.clientX;
    endY = e.clientY;
    
    // Prevent default behaviors
    e.preventDefault();
    e.stopPropagation();
    
    // Initial update
    updateMeasurement(endX, endY);
  }
  
  // Mouse move handler
  function onMouseMove(e) {
    if (!isActive || !isDrawing) return;
    
    // Update measurement
    updateMeasurement(e.clientX, e.clientY);
    
    // Prevent default behaviors
    e.preventDefault();
    e.stopPropagation();
  }
  
  // Mouse up handler
  function onMouseUp(e) {
    if (!isActive || !isDrawing) return;
    
    // Finalize measurement
    updateMeasurement(e.clientX, e.clientY);
    
    // Prevent default behaviors
    e.preventDefault();
    e.stopPropagation();
    
    // End drawing
    isDrawing = false;
    
    // Calculate final dimensions
    const width = Math.abs(endX - startX);
    const height = Math.abs(endY - startY);
    const diagonal = Math.sqrt(width * width + height * height);
    
    // Save measurement
    const measurements = {
      width: width,
      height: height,
      diagonal: Math.round(diagonal)
    };
    
    console.log('Measurement taken:', measurements);
    
    // Save to storage
    chrome.storage.local.set({lastMeasurement: measurements}, function() {
      console.log('Measurement saved to storage');
    });
    
    // Send message to background
    chrome.runtime.sendMessage({
      action: 'measurementTaken',
      measurements: measurements
    });
    
    // Show confirmation in info panel
    infoPanel.innerHTML = `
      <div style="text-align:center;padding:20px;">
        <div style="color:#10B981;font-size:24px;margin-bottom:8px;">✓</div>
        <div style="font-weight:600;margin-bottom:8px;">Measurement Saved!</div>
        <div style="font-size:14px;">
          <span style="color:#9CA3AF;">W:</span> ${width}px
          <span style="color:#9CA3AF;margin-left:12px;">H:</span> ${height}px
        </div>
      </div>
    `;
    
    // Clean up after delay
    setTimeout(cleanup, 1500);
  }
  
  // Escape key handler
  function onKeyDown(e) {
    if (e.key === 'Escape' && isActive) {
      console.log('Measurement cancelled with Escape key');
      cleanup();
    }
  }
  
  // Cleanup function
  function cleanup() {
    if (!isActive) return;
    
    console.log('Cleaning up measurement tool');
    isActive = false;
    isDrawing = false;
    
    // Remove event listeners
    overlay.removeEventListener('mousedown', onMouseDown);
    overlay.removeEventListener('mousemove', onMouseMove);
    overlay.removeEventListener('mouseup', onMouseUp);
    document.removeEventListener('keydown', onKeyDown);
    
    // Restore original styles
    document.body.style.cursor = originalCursor;
    document.body.style.userSelect = originalUserSelect;
    
    // Remove UI elements
    document.body.removeChild(overlay);
    document.body.removeChild(infoPanel);
    document.body.removeChild(measureBox);
    document.body.removeChild(widthLabel);
    document.body.removeChild(heightLabel);
    document.body.removeChild(diagonalLabel);
    
    console.log('Measurement tool cleanup complete');
  }
  
  // Add event listeners
  overlay.addEventListener('mousedown', onMouseDown);
  overlay.addEventListener('mousemove', onMouseMove);
  overlay.addEventListener('mouseup', onMouseUp);
  document.addEventListener('keydown', onKeyDown);
  
  console.log('Measurement tool event listeners added');
}

// Helper function to convert RGB to HEX
function rgbToHex(rgb) {
  // Check if already a hex color
  if (rgb.startsWith('#')) {
    return rgb.toUpperCase();
  }
  
  try {
    // Parse RGB values
    const rgbMatch = rgb.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)(?:,\s*[\d.]+)?\)/i);
    
    if (!rgbMatch) {
      console.error('Failed to parse RGB:', rgb);
      return '#000000'; // Default black if parsing fails
    }
    
    // Convert to hex
    const r = parseInt(rgbMatch[1]);
    const g = parseInt(rgbMatch[2]);
    const b = parseInt(rgbMatch[3]);
    
    const hex = '#' + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1).toUpperCase();
    
    return hex;
  } catch (error) {
    console.error('Error converting RGB to HEX:', error);
    return '#000000'; // Default black if conversion fails
  }
}