// Web Design Ruler - Popup Script

// Import palette functions - these are defined in palette.js which should be loaded before this script

// Global variables
let currentPaletteName = null;
let lastPickedColor = null;
let palettes = {};

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
  console.log('Web Design Ruler popup loaded');
  
  // Initialize tab functionality
  initializeTabs();
  
  // Initialize tool buttons
  initializeToolButtons();
  
  // Initialize palette functionality
  initializePalettes();
  
  // Load stored data
  loadStoredData();
  
  // Initialize copy buttons
  initializeCopyButtons();
});

// Tab switching functionality
function initializeTabs() {
  const tabButtons = document.querySelectorAll('.tab-btn');
  const tabPanes = document.querySelectorAll('.tab-pane');
  
  tabButtons.forEach(button => {
    button.addEventListener('click', () => {
      // Remove active class from all buttons and panes
      tabButtons.forEach(btn => btn.classList.remove('active'));
      tabPanes.forEach(pane => pane.classList.remove('active'));
      
      // Add active class to clicked button and corresponding pane
      button.classList.add('active');
      const tabId = button.getAttribute('data-tab') + '-tab';
      document.getElementById(tabId).classList.add('active');
    });
  });
}

// Initialize tool activation buttons
function initializeToolButtons() {
  // Color picker button
  document.getElementById('color-picker-btn').addEventListener('click', function() {
    activateTool('activateColorPicker', 'color picker');
  });
  
  // Font detector button
  document.getElementById('font-detector-btn').addEventListener('click', function() {
    activateTool('activateFontDetector', 'font detector');
  });
  
  // Measure tool button
  document.getElementById('measure-tool-btn').addEventListener('click', function() {
    activateTool('activateMeasureTool', 'measurement tool');
  });
}

// Generic tool activation function
function activateTool(action, toolName) {
  chrome.runtime.sendMessage(
    {action: action}, 
    function(response) {
      if (chrome.runtime.lastError) {
        console.error('Error:', chrome.runtime.lastError);
        showNotification(`Cannot activate the ${toolName}: ${chrome.runtime.lastError.message}`, 'error');
        return;
      }
      
      if (response && response.success) {
        window.close(); // Close popup if successful
      } else if (response && response.error) {
        showNotification(`Cannot activate the ${toolName}: ${response.error}`, 'error');
      } else {
        showNotification(`Cannot activate the ${toolName}. Try refreshing the page first.`, 'error');
      }
    }
  );
}

// Load stored data from Chrome storage
function loadStoredData() {
  // Load last picked color
  chrome.storage.local.get('lastPickedColor', function(data) {
    if (data.lastPickedColor) {
      displayPickedColor(data.lastPickedColor);
    }
  });
  
  // Load recent colors
  chrome.storage.local.get('recentColors', function(data) {
    if (data.recentColors && data.recentColors.length > 0) {
      displayRecentColors(data.recentColors);
    }
  });
  
  // Load last detected font
  chrome.storage.local.get('lastDetectedFont', function(data) {
    if (data.lastDetectedFont) {
      displayFontDetails(data.lastDetectedFont);
    }
  });
  
  // Load last measurement
  chrome.storage.local.get('lastMeasurement', function(data) {
    if (data.lastMeasurement) {
      displayMeasurement(data.lastMeasurement);
    }
  });
}

// Display picked color
function displayPickedColor(color) {
  lastPickedColor = color;
  
  const colorResult = document.getElementById('color-result');
  const colorSwatch = document.getElementById('color-swatch');
  const colorHex = document.getElementById('color-hex');
  const colorRgb = document.getElementById('color-rgb');
  const colorHsl = document.getElementById('color-hsl');
  
  colorResult.classList.remove('hidden');
  colorSwatch.style.backgroundColor = color;
  
  // Convert hex to other formats
  const rgb = hexToRgb(color);
  const hsl = rgbToHsl(rgb.r, rgb.g, rgb.b);
  
  colorHex.value = color;
  colorRgb.value = `rgb(${rgb.r}, ${rgb.g}, ${rgb.b})`;
  colorHsl.value = `hsl(${hsl.h}, ${hsl.s}%, ${hsl.l}%)`;
  
  // Set up save to palette button
  document.getElementById('save-to-palette').onclick = () => {
    if (currentPaletteName) {
      saveToPalette(currentPaletteName, color, () => {
        loadPalettes(updatePaletteDisplay);
        showNotification('Color saved to palette!', 'success');
      });
    } else {
      showNotification('Please select or create a palette first', 'warning');
      // Switch to palettes tab
      document.querySelector('[data-tab="palettes"]').click();
    }
  };
}

// Display recent colors
function displayRecentColors(colors) {
  const recentColorsSection = document.getElementById('recent-colors');
  const recentColorsGrid = document.getElementById('recent-colors-grid');
  
  recentColorsSection.classList.remove('hidden');
  recentColorsGrid.innerHTML = '';
  
  colors.forEach(color => {
    const swatch = createColorSwatch(color);
    swatch.addEventListener('click', () => displayPickedColor(color));
    recentColorsGrid.appendChild(swatch);
  });
}

// Display font details
function displayFontDetails(fontDetails) {
  const fontResult = document.getElementById('font-result');
  const fontPreview = document.getElementById('font-preview');
  const fontProperties = document.getElementById('font-properties');
  
  fontResult.classList.remove('hidden');
  
  // Set preview text
  fontPreview.style.fontFamily = fontDetails.fontFamily;
  fontPreview.style.fontSize = fontDetails.fontSize;
  fontPreview.style.fontWeight = fontDetails.fontWeight.split(' ')[0];
  fontPreview.style.fontStyle = fontDetails.fontStyle;
  fontPreview.style.color = fontDetails.color;
  fontPreview.textContent = 'The quick brown fox jumps over the lazy dog';
  
  // Display properties
  fontProperties.innerHTML = '';
  const properties = [
    { name: 'Font Family', value: fontDetails.fontFamily.replace(/['"]/g, '') },
    { name: 'Size', value: fontDetails.fontSize },
    { name: 'Weight', value: fontDetails.fontWeight },
    { name: 'Style', value: fontDetails.fontStyle },
    { name: 'Line Height', value: fontDetails.lineHeight },
    { name: 'Letter Spacing', value: fontDetails.letterSpacing },
    { name: 'Color', value: fontDetails.color }
  ];
  
  properties.forEach(prop => {
    const li = document.createElement('li');
    li.innerHTML = `
      <span class="property-name">${prop.name}:</span>
      <span>${prop.value}</span>
    `;
    fontProperties.appendChild(li);
  });
  
  // Set up copy CSS button
  document.getElementById('copy-font-css').onclick = () => {
    copyToClipboard(fontDetails.css);
    showNotification('CSS copied to clipboard!', 'success');
  };
  
  // Set up save font button
  document.getElementById('save-font').onclick = () => {
    // Save font details to storage for later reference
    chrome.storage.local.get('savedFonts', ({ savedFonts = [] }) => {
      savedFonts.unshift({
        ...fontDetails,
        savedAt: new Date().toISOString()
      });
      // Keep only last 50 saved fonts
      savedFonts = savedFonts.slice(0, 50);
      chrome.storage.local.set({ savedFonts }, () => {
        showNotification('Font saved!', 'success');
      });
    });
  };
}

// Display measurement
function displayMeasurement(measurement) {
  const measureResult = document.getElementById('measure-result');
  const measureWidth = document.getElementById('measure-width');
  const measureHeight = document.getElementById('measure-height');
  const measureDiagonal = document.getElementById('measure-diagonal');
  const measurementVisual = document.getElementById('measurement-visual');
  
  measureResult.classList.remove('hidden');
  
  measureWidth.value = measurement.width;
  measureHeight.value = measurement.height;
  measureDiagonal.value = measurement.diagonal;
  
  // Create visual representation
  measurementVisual.innerHTML = '';
  
  // Calculate scaled size to fit in the container
  const containerWidth = 340; // Approximate width
  const containerHeight = 120;
  
  const scale = Math.min(
    (containerWidth - 40) / measurement.width,
    (containerHeight - 40) / measurement.height,
    1
  );
  
  const scaledWidth = Math.max(measurement.width * scale, 20);
  const scaledHeight = Math.max(measurement.height * scale, 20);
  
  // Create visual box
  const visualBox = document.createElement('div');
  visualBox.style.position = 'absolute';
  visualBox.style.left = '50%';
  visualBox.style.top = '50%';
  visualBox.style.transform = 'translate(-50%, -50%)';
  visualBox.style.width = scaledWidth + 'px';
  visualBox.style.height = scaledHeight + 'px';
  visualBox.style.border = '2px solid var(--primary-color)';
  visualBox.style.backgroundColor = 'rgba(37, 99, 235, 0.1)';
  visualBox.style.borderRadius = '4px';
  
  measurementVisual.appendChild(visualBox);
}

// Initialize palette functionality
function initializePalettes() {
  // Load palettes
  loadPalettes(updatePaletteDisplay);
  
  // Create new palette button
  document.getElementById('create-palette').addEventListener('click', () => {
    const name = prompt('Enter palette name:');
    if (name && name.trim()) {
      createPalette(name.trim(), [], () => {
        loadPalettes(updatePaletteDisplay);
        currentPaletteName = name.trim();
        document.getElementById('palette-selector').value = currentPaletteName;
        displayPalette(currentPaletteName);
        showNotification('Palette created!', 'success');
      });
    }
  });
  
  // Palette selector
  document.getElementById('palette-selector').addEventListener('change', (e) => {
    currentPaletteName = e.target.value;
    if (currentPaletteName) {
      displayPalette(currentPaletteName);
    } else {
      document.getElementById('current-palette').classList.add('hidden');
    }
  });
  
  // Import palette
  document.getElementById('import-palette').addEventListener('change', (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        try {
          const data = JSON.parse(event.target.result);
          if (data.name && Array.isArray(data.colors)) {
            createPalette(data.name, data.colors, () => {
              loadPalettes(updatePaletteDisplay);
              showNotification('Palette imported!', 'success');
            });
          } else {
            showNotification('Invalid palette file', 'error');
          }
        } catch (error) {
          showNotification('Error importing palette', 'error');
        }
      };
      reader.readAsText(file);
    }
  });
}

// Update palette display
function updatePaletteDisplay(loadedPalettes) {
  palettes = loadedPalettes;
  const selector = document.getElementById('palette-selector');
  
  // Clear existing options
  selector.innerHTML = '<option value="">Select a palette...</option>';
  
  // Add palette options
  Object.keys(palettes).forEach(name => {
    const option = document.createElement('option');
    option.value = name;
    option.textContent = name;
    selector.appendChild(option);
  });
  
  // Restore selected palette if it still exists
  if (currentPaletteName && palettes[currentPaletteName]) {
    selector.value = currentPaletteName;
    displayPalette(currentPaletteName);
  }
}

// Display a specific palette
function displayPalette(paletteName) {
  const palette = palettes[paletteName];
  if (!palette) return;
  
  const paletteContainer = document.getElementById('current-palette');
  const paletteNameEl = document.getElementById('palette-name');
  const paletteColors = document.getElementById('palette-colors');
  
  paletteContainer.classList.remove('hidden');
  paletteNameEl.textContent = paletteName;
  
  // Clear existing colors
  paletteColors.innerHTML = '';
  
  // Display colors
  palette.forEach(color => {
    const swatch = createColorSwatch(color);
    swatch.addEventListener('click', () => displayPickedColor(color));
    swatch.addEventListener('contextmenu', (e) => {
      e.preventDefault();
      if (confirm(`Remove ${color} from palette?`)) {
        removeFromPalette(paletteName, color, () => {
          loadPalettes(updatePaletteDisplay);
          showNotification('Color removed from palette', 'success');
        });
      }
    });
    paletteColors.appendChild(swatch);
  });
  
  // Set up palette actions
  document.getElementById('edit-palette-name').onclick = () => {
    const newName = prompt('Enter new palette name:', paletteName);
    if (newName && newName.trim() && newName !== paletteName) {
      renamePalette(paletteName, newName.trim(), (success) => {
        if (success) {
          currentPaletteName = newName.trim();
          loadPalettes(updatePaletteDisplay);
          showNotification('Palette renamed!', 'success');
        } else {
          showNotification('Could not rename palette', 'error');
        }
      });
    }
  };
  
  document.getElementById('export-palette').onclick = () => {
    const data = {
      name: paletteName,
      colors: palette
    };
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${paletteName.replace(/[^a-z0-9]/gi, '_')}_palette.json`;
    a.click();
    URL.revokeObjectURL(url);
    showNotification('Palette exported!', 'success');
  };
  
  document.getElementById('delete-palette').onclick = () => {
    if (confirm(`Delete palette "${paletteName}"?`)) {
      deletePalette(paletteName, () => {
        currentPaletteName = null;
        loadPalettes(updatePaletteDisplay);
        document.getElementById('current-palette').classList.add('hidden');
        showNotification('Palette deleted', 'success');
      });
    }
  };
}

// Create color swatch element
function createColorSwatch(color) {
  const swatch = document.createElement('div');
  swatch.className = 'color-swatch-small';
  swatch.style.backgroundColor = color;
  swatch.setAttribute('data-color', color);
  swatch.title = color;
  return swatch;
}

// Initialize copy buttons
function initializeCopyButtons() {
  document.querySelectorAll('.copy-btn').forEach(btn => {
    btn.addEventListener('click', function() {
      const valueType = this.getAttribute('data-value');
      let value = '';
      
      switch(valueType) {
        case 'hex':
          value = document.getElementById('color-hex').value;
          break;
        case 'rgb':
          value = document.getElementById('color-rgb').value;
          break;
        case 'hsl':
          value = document.getElementById('color-hsl').value;
          break;
      }
      
      if (value) {
        copyToClipboard(value);
        const originalText = this.textContent;
        this.textContent = 'Copied!';
        setTimeout(() => {
          this.textContent = originalText;
        }, 1000);
      }
    });
  });
}

// Utility Functions
function copyToClipboard(text) {
  const textarea = document.createElement('textarea');
  textarea.value = text;
  document.body.appendChild(textarea);
  textarea.select();
  document.execCommand('copy');
  document.body.removeChild(textarea);
}

function showNotification(message, type = 'info') {
  // Create notification element
  const notification = document.createElement('div');
  notification.className = `notification ${type}`;
  notification.textContent = message;
  notification.style.cssText = `
    position: fixed;
    bottom: 20px;
    left: 50%;
    transform: translateX(-50%);
    padding: 12px 20px;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 500;
    z-index: 10000;
    animation: slideUp 0.3s ease;
  `;
  
  // Set colors based on type
  switch(type) {
    case 'success':
      notification.style.backgroundColor = '#10B981';
      notification.style.color = 'white';
      break;
    case 'error':
      notification.style.backgroundColor = '#EF4444';
      notification.style.color = 'white';
      break;
    case 'warning':
      notification.style.backgroundColor = '#F59E0B';
      notification.style.color = 'white';
      break;
    default:
      notification.style.backgroundColor = '#374151';
      notification.style.color = 'white';
  }
  
  document.body.appendChild(notification);
  
  // Remove after 3 seconds
  setTimeout(() => {
    notification.style.animation = 'slideDown 0.3s ease';
    setTimeout(() => {
      document.body.removeChild(notification);
    }, 300);
  }, 3000);
}

// Color conversion functions
function hexToRgb(hex) {
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? {
    r: parseInt(result[1], 16),
    g: parseInt(result[2], 16),
    b: parseInt(result[3], 16)
  } : null;
}

function rgbToHsl(r, g, b) {
  r /= 255;
  g /= 255;
  b /= 255;
  
  const max = Math.max(r, g, b);
  const min = Math.min(r, g, b);
  let h, s, l = (max + min) / 2;
  
  if (max === min) {
    h = s = 0;
  } else {
    const d = max - min;
    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
    
    switch (max) {
      case r: h = (g - b) / d + (g < b ? 6 : 0); break;
      case g: h = (b - r) / d + 2; break;
      case b: h = (r - g) / d + 4; break;
    }
    
    h /= 6;
  }
  
  return {
    h: Math.round(h * 360),
    s: Math.round(s * 100),
    l: Math.round(l * 100)
  };
}

// Add CSS animation keyframes
const style = document.createElement('style');
style.textContent = `
  @keyframes slideUp {
    from { transform: translate(-50%, 100%); opacity: 0; }
    to { transform: translate(-50%, 0); opacity: 1; }
  }
  @keyframes slideDown {
    from { transform: translate(-50%, 0); opacity: 1; }
    to { transform: translate(-50%, 100%); opacity: 0; }
  }
`;
document.head.appendChild(style);

// Listen for messages from background script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Popup received message:', message);
  
  if (message.action === 'colorPicked' && message.color) {
    displayPickedColor(message.color);
  } else if (message.action === 'fontDetected' && message.fontDetails) {
    displayFontDetails(message.fontDetails);
  } else if (message.action === 'measurementTaken' && message.measurements) {
    displayMeasurement(message.measurements);
  }
});