/**
 * Web Design Ruler Palette Management
 * Provides functions to manage color palettes
 */

/**
 * Saves a color to a palette
 * @param {string} paletteName - Name of palette
 * @param {string} color - Color in hex format
 * @param {Function} callback - Callback function when done
 */
function saveToPalette(paletteName, color, callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    // Create palette if it doesn't exist
    if (!palettes[paletteName]) {
      palettes[paletteName] = [];
    }
    
    // Add color to palette if it doesn't already exist
    if (!palettes[paletteName].includes(color)) {
      palettes[paletteName].push(color);
    }
    
    // Save palettes
    chrome.storage.local.set({ palettes }, callback);
  });
}

/**
 * Creates a new palette
 * @param {string} paletteName - Name of palette
 * @param {Array} colors - Initial colors
 * @param {Function} callback - Callback function when done
 */
function createPalette(paletteName, colors = [], callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    // Create new palette
    palettes[paletteName] = colors;
    
    // Save palettes
    chrome.storage.local.set({ palettes }, callback);
  });
}

/**
 * Deletes a palette
 * @param {string} paletteName - Name of palette to delete
 * @param {Function} callback - Callback function when done
 */
function deletePalette(paletteName, callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    // Delete palette
    delete palettes[paletteName];
    
    // Save palettes
    chrome.storage.local.set({ palettes }, callback);
  });
}

/**
 * Loads a list of all palettes
 * @param {Function} callback - Callback function with palettes as parameter
 */
function loadPalettes(callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    callback(palettes);
  });
}

/**
 * Renames a palette
 * @param {string} oldName - Current palette name
 * @param {string} newName - New palette name
 * @param {Function} callback - Callback function when done
 */
function renamePalette(oldName, newName, callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    // Return if palette doesn't exist or new name is taken
    if (!palettes[oldName] || palettes[newName]) {
      callback(false);
      return;
    }
    
    // Create new palette with same colors
    palettes[newName] = palettes[oldName];
    
    // Delete old palette
    delete palettes[oldName];
    
    // Save palettes
    chrome.storage.local.set({ palettes }, () => callback(true));
  });
}

/**
 * Removes a color from a palette
 * @param {string} paletteName - Name of palette
 * @param {string} color - Color to remove
 * @param {Function} callback - Callback function when done
 */
function removeFromPalette(paletteName, color, callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    // Return if palette doesn't exist
    if (!palettes[paletteName]) {
      callback(false);
      return;
    }
    
    // Remove color from palette
    palettes[paletteName] = palettes[paletteName].filter(c => c !== color);
    
    // Save palettes
    chrome.storage.local.set({ palettes }, () => callback(true));
  });
}

/**
 * Gets a specific palette
 * @param {string} paletteName - Name of palette
 * @param {Function} callback - Callback function with palette as parameter
 */
function getPalette(paletteName, callback) {
  chrome.storage.local.get('palettes', ({ palettes = {} }) => {
    callback(palettes[paletteName] || null);
  });
}