/**
 * Web Design Ruler Background Script
 * Handles background processes and communication between 
 * content scripts and popup
 */

console.log('Web Design Ruler background worker started');

// Initialize palette storage when extension is installed
chrome.runtime.onInstalled.addListener(() => {
  console.log('Web Design Ruler installed');
  
  // Create default palettes with neutral design colors
  const defaultPalettes = {
    'LXB Studio': [
      '#256EFF', // LXB Primary Blue
      '#34314C', // LXB Secondary Dark
      '#47E5BC', // LXB Accent Teal
      '#F8F9FA', // Light
      '#212529'  // Dark
    ],
    'Material Design': [
      '#2196F3', // Blue
      '#FF5722', // Deep Orange
      '#4CAF50', // Green
      '#FFC107', // Amber
      '#9C27B0'  // Purple
    ],
    'Neutrals': [
      '#000000', // Black
      '#424242', // Dark Gray
      '#757575', // Gray
      '#BDBDBD', // Light Gray
      '#FFFFFF'  // White
    ]
  };
  
  // Initialize storage
  chrome.storage.local.get('palettes', ({ palettes }) => {
    if (!palettes) {
      chrome.storage.local.set({ palettes: defaultPalettes });
    }
  });
  
  // Initialize recent colors
  chrome.storage.local.get('recentColors', ({ recentColors }) => {
    if (!recentColors) {
      chrome.storage.local.set({ recentColors: [] });
    }
  });
  
  // Add context menu items
  try {
    // Color Picker
    chrome.contextMenus.create({
      id: 'wdr-eyedropper',
      title: 'Pick Color with Web Design Ruler',
      contexts: ['page']
    });
    
    // Font Detector
    chrome.contextMenus.create({
      id: 'wdr-font-detector',
      title: 'Identify Font with Web Design Ruler',
      contexts: ['page']
    });
    
    // Measure Tool
    chrome.contextMenus.create({
      id: 'wdr-measure-tool',
      title: 'Measure with Web Design Ruler',
      contexts: ['page']
    });
  } catch (error) {
    console.error('Error creating context menus:', error);
  }
});

// Context menu click handler
chrome.contextMenus.onClicked.addListener((info, tab) => {
  console.log('Context menu clicked:', info.menuItemId);
  
  let actionType = "";
  
  // Map menu IDs to action types
  switch (info.menuItemId) {
    case 'wdr-eyedropper':
      actionType = "activateColorPicker";
      break;
    case 'wdr-font-detector':
      actionType = "activateFontDetector";
      break;
    case 'wdr-measure-tool':
      actionType = "activateMeasureTool";
      break;
    default:
      console.error('Unknown menu item ID:', info.menuItemId);
      return;
  }
  
  // Use standard activation process
  activateTool(actionType);
});

// General tool activation function
function activateTool(actionType) {
  chrome.tabs.query({active: true, currentWindow: true}, async (tabs) => {
    if (!tabs || tabs.length === 0) {
      console.error('No active tab found');
      return;
    }
    
    const tab = tabs[0];
    const url = tab.url;
    
    // Check if this is a valid URL for content script
    if (url.startsWith('chrome://') || url.startsWith('chrome-extension://') || url === 'about:blank') {
      console.error('Cannot use tool on browser pages');
      return;
    }
    
    try {
      // First, check if content script is already there
      console.log('Checking if content script is available...');
      let contentScriptReady = false;
      
      try {
        // Try to ping the content script
        await new Promise((resolve, reject) => {
          const pingTimeout = setTimeout(() => {
            reject(new Error('Ping timed out'));
          }, 500);
          
          chrome.tabs.sendMessage(tab.id, {action: "ping"}, response => {
            clearTimeout(pingTimeout);
            if (response && response.pong) {
              contentScriptReady = true;
              resolve();
            } else {
              reject(new Error('No pong response'));
            }
          });
        });
      } catch (pingError) {
        console.log('Content script not detected, will inject it now');
      }
      
      // If not ready, inject the script
      if (!contentScriptReady) {
        console.log('Injecting content script...');
        await chrome.scripting.executeScript({
          target: {tabId: tab.id},
          files: ['scripts/content-script.js']
        });
        
        // Wait a bit for the script to initialize
        await new Promise(resolve => setTimeout(resolve, 100));
      }
      
      // Now activate the requested tool
      console.log(`Sending ${actionType} message...`);
      chrome.tabs.sendMessage(tab.id, {action: actionType});
      
    } catch (error) {
      console.error('Error:', error);
    }
  });
}

// Listen for messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Background received message:', message);
  
  // Message relay from content script to popup
  if (['colorPicked', 'fontDetected', 'measurementTaken'].includes(message.action)) {
    // Forward to popup if it's open
    chrome.runtime.sendMessage(message)
      .catch(() => {
        // Popup is not open, store data for later
        if (message.action === 'colorPicked') {
          chrome.storage.local.set({ lastPickedColor: message.color });
          
          // Add to recent colors
          chrome.storage.local.get('recentColors', ({ recentColors = [] }) => {
            // Remove duplicates and keep only last 20
            recentColors = recentColors.filter(c => c !== message.color);
            recentColors.unshift(message.color);
            recentColors = recentColors.slice(0, 20);
            chrome.storage.local.set({ recentColors });
          });
        } else if (message.action === 'fontDetected') {
          chrome.storage.local.set({ lastDetectedFont: message.fontDetails });
        } else if (message.action === 'measurementTaken') {
          chrome.storage.local.set({ lastMeasurement: message.measurements });
        }
      });
  }
  
  // Handle color picker activation
  if (message.action === "activateColorPicker") {
    activateTool("activateColorPicker");
    sendResponse({success: true});
    return true;
  }
  
  // Handle font detector activation
  if (message.action === "activateFontDetector") {
    activateTool("activateFontDetector");
    sendResponse({success: true});
    return true;
  }
  
  // Handle measure tool activation
  if (message.action === "activateMeasureTool") {
    activateTool("activateMeasureTool");
    sendResponse({success: true});
    return true;
  }
  
  return true; // Keep the message channel open for asynchronous response
});

// Handle keyboard shortcuts
chrome.commands.onCommand.addListener((command) => {
  if (command === 'activate_eyedropper') {
    activateTool("activateColorPicker");
  }
});